using System;
using System.IO;
using System.Text.RegularExpressions;
using System.Drawing;
using System.Drawing.Imaging;
using System.Diagnostics;

namespace WOWMapCreator
{
	public class WOWMapCreator
	{

		/// <summary>
		/// 
		/// </summary>
		/// <param name="fileName">Name of TRS File to parse</param>
		/// <param name="rootFolder">Folder where all of the BLP files are</param>
		public void MoveImagesToFolders(string fileName, string rootFolder)
		{
			StreamReader reader = File.OpenText(fileName);
			string line;
			while ( (line = reader.ReadLine()) != null )
			{
				if (line.Substring(0, 5) == "dir: ")
				{
					string dirName = line.Substring(5);
					Console.WriteLine("Create Dir {0}\\{1}", rootFolder, dirName);
					Directory.CreateDirectory(String.Format(@"{0}\{1}", rootFolder, dirName));
				}
				else
				{
					Match match = Regex.Match(line, "(?<DestFile>[^\t]+)\t(?<SrcFile>[^\t]+)");
					string srcFile = match.Groups["SrcFile"].Value;
					string destFile = match.Groups["DestFile"].Value;

					Console.WriteLine("Move {0} -> {1}", rootFolder+"\\"+srcFile, rootFolder+"\\"+destFile);
					if (File.Exists(rootFolder+"\\"+srcFile))
						File.Move(rootFolder+"\\"+srcFile, rootFolder+"\\"+destFile);
					else
						Console.WriteLine("Couldn't find {0}", rootFolder+"\\"+srcFile);
				}
			}
		}

		public void ConvertBLPToTGA(params string[] files)
		{
			for (int i = 0; i < files.Length; ++i)
			{
				string tgaFileName = Path.ChangeExtension(files[i], ".tga");
				if ( !File.Exists(tgaFileName) )
				{
					Process exeProcess = new Process();
					exeProcess.StartInfo.FileName = @"BLP2toTGA.exe";
					exeProcess.StartInfo.Arguments = String.Format("\"{0}\"", files[i]);
					exeProcess.StartInfo.UseShellExecute = false;
					exeProcess.StartInfo.WindowStyle = System.Diagnostics.ProcessWindowStyle.Minimized;
					exeProcess.StartInfo.RedirectStandardInput = true;
					exeProcess.StartInfo.CreateNoWindow = false;
					exeProcess.Start();

					// Send a Enter so it will end
					exeProcess.StandardInput.Write(exeProcess.StandardInput.NewLine);

					// Wait for it to finish
					exeProcess.WaitForExit();
				}
			}
		}

		public void ConvertTGAToBMP(params string[] files)
		{
			foreach (string srcFile in files)
			{
				string destFile = Path.ChangeExtension(srcFile, ".bmp");

				if (!File.Exists(destFile))
				{
					uint hImage = FreeImageAPI.FreeImage.Load(FreeImageAPI.FIF.FIF_TARGA, srcFile, 0);
					FreeImageAPI.FreeImage.Save(FreeImageAPI.FIF.FIF_BMP, hImage, destFile, 0);
				}
			}
		}

		public void ConvertBMPToTGA(params string[] files)
		{
			foreach (string srcFile in files)
			{
				string destFile = Path.ChangeExtension(srcFile, ".tga");

				uint hImage = FreeImageAPI.FreeImage.Load(FreeImageAPI.FIF.FIF_BMP, srcFile, 0);
				FreeImageAPI.FreeImage.Save(FreeImageAPI.FIF.FIF_TARGA, hImage, destFile, 0);
			}
		}

		public Image MakeMap(float scale, int leftOffset, int topOffset, int rightOffset, int bottomOffset, params string[] files)
		{
			if (files.Length == 0)
				return null;

			Rectangle mapDimensions = GetImageDimensions(files);
			mapDimensions.X += leftOffset;
			mapDimensions.Y += topOffset;
			mapDimensions.Width -= leftOffset;
			mapDimensions.Height -= topOffset;
			mapDimensions.Width += rightOffset;
			mapDimensions.Height += bottomOffset;

			// Get the Image Width and Height
			int imgWidth = 256;
			int imgHeight = 256;

			int scaleImgWidth = (int)(scale * imgWidth);
			int scaleImgHeight = (int)(scale * imgHeight);

			int destImgWidth = (int)(scale * mapDimensions.Width * imgWidth);
			int destImgHeight = (int)(scale * mapDimensions.Height * imgHeight);

			// Create the New Image
			Image tImage = Image.FromFile(files[0]);
			Image destImage = new Bitmap(tImage, destImgWidth, destImgHeight);
			Graphics g = Graphics.FromImage(destImage);
			g.Clear(Color.Black);

			foreach (string file in files)
			{
				// If we can parse its position
				int column, row;
				if (GetFilePosition(file, out column, out row))
				{
					// Load the Image
					Image img = Image.FromFile(file);
					int x = (int)((column - mapDimensions.Left) * scaleImgWidth);
					int y = (int)((row - mapDimensions.Top) * scaleImgHeight);

					// Draw it in its position
					g.DrawImage(img, x, y, (int)(scale*img.Width), (int)(scale*img.Width));
				}
			}

			return destImage;
		}

		
		Rectangle GetImageDimensions(params string[] files)
		{
			int columnMin = int.MaxValue;
			int columnMax = int.MinValue;
			int rowMin = int.MaxValue;
			int rowMax = int.MinValue;
			foreach (string file in files)
			{
				int column, row;
				if (GetFilePosition(file, out column, out row))
				{
					if (column < columnMin)	columnMin = column;
					if (column > columnMax)	columnMax = column;
					if (row < rowMin)		rowMin = row;
					if (row > rowMax)		rowMax = row;
				}
			}

			return Rectangle.FromLTRB(columnMin, rowMin, columnMax, rowMax);
		}

		bool GetFilePosition(string fileName, out int column, out int row)
		{
			column = 0;
			row = 0;
			if (Path.GetFileName(fileName).StartsWith("map"))
			{

				Match match = Regex.Match(fileName, @"map(?<Column>[0-9]{2})_(?<Row>[0-9]{2})");

				if (match != null)
				{
					column = int.Parse(match.Groups["Column"].Value);
					row = int.Parse(match.Groups["Row"].Value);
					return true;
				}
				else
				{
					column = 0;
					row = 0;
					Console.WriteLine("Unable to parse for coordinates {0}", fileName);
					return false;
				}
			}
			return false;
		}
	}
}
