{ -----------------------------------------------------------------------
Tcl Scripting Language Components (Tslc)
Copyright (C) 1996-2002 William Byrne

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

WilliamB@ByrneLitho.com
------------------------------------------------------------------------}
unit TslcEdit;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, DsgnIntf, Buttons, ExtCtrls, ComCtrls, Tslc, TslcEmul;

const
	WM_NAMECHANGE = 2048;

type
  TTclEditForm = class(TForm)
    PageControl: TPageControl;
    SheetCommands: TTabSheet;
    SheetSwitches: TTabSheet;
    SheetParams: TTabSheet;
    SheetTraces: TTabSheet;
    ListCommands: TListBox;
    ListSwitches: TListBox;
    ListParams: TListBox;
    ListTraces: TListBox;
    BtnInsertSwitch: TBitBtn;
    BtnDeleteSwitch: TBitBtn;
    BtnInsertCommand: TBitBtn;
    BtnDeleteCommand: TBitBtn;
    BtnInsertParam: TBitBtn;
    BtnDeleteParam: TBitBtn;
    BtnInsertTrace: TBitBtn;
    BtnDeleteTrace: TBitBtn;
    BtnHelp: TButton;
    LabelCommand: TLabel;
    LabelSwitches: TLabel;
    LabelParameters: TLabel;
    BtnEmulate: TButton;
    SheetChannels: TTabSheet;
    BtnInsertChannel: TBitBtn;
    ListChannels: TListBox;
    BtnRemoveChannel: TBitBtn;
    ListAvailChannels: TListBox;
    Label1: TLabel;
    Label2: TLabel;
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormCreate(Sender: TObject);
    procedure BtnInsertCommandClick(Sender: TObject);
    procedure BtnInsertTraceClick(Sender: TObject);
    procedure BtnDeleteCommandClick(Sender: TObject);
    procedure BtnDeleteTraceClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ListCommandsClick(Sender: TObject);
    procedure BtnInsertSwitchClick(Sender: TObject);
    procedure BtnDeleteSwitchClick(Sender: TObject);
    procedure ListTracesClick(Sender: TObject);
    procedure ListSwitchesClick(Sender: TObject);
    procedure ListParamsClick(Sender: TObject);
    procedure BtnInsertParamClick(Sender: TObject);
    procedure BtnDeleteParamClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure BtnHelpClick(Sender: TObject);
    procedure PageControlChange(Sender: TObject);
    procedure BtnEmulateClick(Sender: TObject);
    procedure ListCommandsDblClick(Sender: TObject);
    procedure ListCommandsMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ListCommandsDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure ListCommandsDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure ListSwitchesDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure ListSwitchesDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure ListSwitchesMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ListParamsDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure ListParamsDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure ListParamsMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ListTracesDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure ListTracesDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure ListTracesMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure BtnInsertChannelClick(Sender: TObject);
    procedure BtnRemoveChannelClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);
  private
    { Private declarations }
    FNewDesigner: TFormDesigner;
    FTcl: TTcl;
    FOnTop: boolean;
    FPrevHelp: THelpEvent;
    function GetSelectedCommand: TTclCommand;
    procedure NameChange( comp: TComponent; const NewName: TComponentName );
    procedure RefreshLists;
    procedure RefreshChannelLists;
    procedure Select(AComponent: TComponent);
    procedure WMSysCommand(var Message: TWMSysCommand); message WM_SYSCOMMAND;
    procedure WMNameChange(var Message: TMessage); message WM_NAMECHANGE;
    procedure UpdateCommandButtons(enable: boolean);
    procedure UpdateCommandInfo(cmd: TTclCommand);
    procedure DeleteListItem(itemIndex: integer; items: TStrings; comp: TComponent);
    procedure DoHelp(ctx: string);
    function CurrentHelpTopic: string;
  public
    { Public declarations }
    procedure Enter(ADesigner: TFormDesigner; ATcl: TTcl);
    destructor Destroy; override;
  end;

procedure ShowTclDesigner(ADesigner: TFormDesigner; ATcl: TTcl);
implementation
//uses DsgnWnds, TslcEmul;
{$D-,I-,Y-}
{$R *.DFM}

const
	SC_STAYONTOP = $F00;


procedure ShowTclDesigner(ADesigner: TFormDesigner; ATcl: TTcl);
begin
	if ATcl.Designer <> nil then
    	TTclEditForm(ATcl.Designer).SetFocus
    else
		(TTclEditForm.Create(Application)).Enter(ADesigner, ATcl);
end;

destructor TTclEditForm.Destroy;
begin
	inherited Destroy;
end;

procedure TTclEditForm.WMSysCommand(var Message: TWMSysCommand);
begin
  if (Message.CmdType and $FFF0 = SC_STAYONTOP) then
  begin
  	if not FOnTop then
    begin
        SetWindowPos(Handle, HWND_TOPMOST, 0, 0, 0, 0, SWP_NOMOVE or SWP_NOSIZE or SWP_NOREDRAW);
		CheckMenuItem(GetSystemMenu(Handle, False), SC_STAYONTOP, MF_BYCOMMAND or MF_CHECKED);
    end else
    begin
        SetWindowPos(Handle, HWND_NOTOPMOST, 0, 0, 0, 0, SWP_NOMOVE or SWP_NOSIZE or SWP_NOREDRAW);
        CheckMenuItem(GetSystemMenu(Handle, False), SC_STAYONTOP, MF_BYCOMMAND or MF_UNCHECKED);
    end;
    FOnTop := not FOnTop;
  end else
    inherited;
end;

procedure TTclEditForm.WMNameChange(var Message: TMessage);
begin
    with PTclNewName( Message.LParam )^ do
   		NameChange( Comp, NewName );
end;

procedure TTclEditForm.NameChange(comp: TComponent; const NewName: TComponentName);
	function ScanList( List: TListBox ): boolean;
	var
    	x: integer;
        sel: boolean;
    begin
    	result := False;
        with List, items do
        	for x := 0 to Count - 1 do
	        	if Objects[x] = comp then
                begin
                	sel := itemIndex = x;
    	        	Strings[x] := NewName;
                    if sel then itemIndex := x;
                    result := True;
                    exit;
                end;
	end;
begin
	if ScanList( ListCommands ) then exit;
    if ScanList( ListSwitches ) then exit;
    if ScanList( ListTraces ) then exit;
    ScanList( ListParams );
end;

procedure TTclEditForm.RefreshChannelLists;
var
	x: integer;
    comp: TComponent;
begin
	ListChannels.Clear;
    ListAvailChannels.Clear;
	with TTcl(FTcl) do
    	for x:= 0 to ChannelCount - 1 do
        	ListChannels.items.addObject(Channels[x].Name, Channels[x]);
    if FTcl.Owner <> nil then
    with FTcl.Owner do
    	for x:= 0 to ComponentCount - 1 do
        begin
			comp := Components[x];
        	if (comp is TTclChannel) and (ListChannels.items.IndexOfObject(comp) < 0) then
				ListAvailChannels.items.addObject(comp.Name, comp);
		end;
end;

procedure TTclEditForm.RefreshLists;
var
	x: integer;
begin
	ListCommands.Clear;
    ListTraces.Clear;
    with FTcl do
    	for x := 0 to CommandCount - 1 do
        	ListCommands.items.addObject(Commands[x].Name, Commands[x]);
	with FTcl do
    	for x := 0 to TraceCount - 1 do
        	ListTraces.items.addObject(Traces[x].Name, Traces[x]);
    RefreshChannelLists;
end;

procedure TTclEditForm.Enter(ADesigner: TFormDesigner; ATcl: TTcl);
begin
	FNewDesigner := ADesigner;
    FTcl := ATcl;
	RefreshLists;
	FTcl.Designer := Self;
    Caption := format('%s.%s', [FTcl.Owner.Name, FTcl.Name]);
	Show;
end;

procedure TTclEditForm.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
	Action := caFree;
end;

procedure TTclEditForm.FormCreate(Sender: TObject);
var
	sysMenu : HMenu;
begin
	FOnTop := False;
	FTcl := nil;
    FNewDesigner := nil;
    UpdateCommandButtons(false);
    UpdateCommandInfo(nil);
    PageControl.ActivePage := SheetCommands;
    sysMenu := GetSystemMenu(Handle, false);
    AppendMenu(sysMenu, MF_STRING, SC_STAYONTOP, 'Stay On Top');
end;

procedure TTclEditForm.Select(AComponent: TComponent);
begin
	FNewDesigner.SelectComponent(AComponent);
end;

procedure TTclEditForm.BtnInsertCommandClick(Sender: TObject);
var
	cmd: TTclCommand;
begin
//    cmd := FTcl.BuildCommand( FNewDesigner );
//    TTclCommand(FNew Designer.CreateComponent(TTclCommand, FTcl, 0, 0, 0, 0));
	cmd := TTclCommand(FNewDesigner.CreateComponent( FTcl.BuildCommand, FTcl, 0, 0, 0, 0));
	if cmd.name = '' then
	    cmd.Name := FNewDesigner.UniqueName(cmd.ClassName);
	cmd.Tcl :=  FTcl;
	with ListCommands do
    begin
    	items.addObject(cmd.Name, cmd);
        itemIndex := items.indexOfObject(cmd);
    end;
end;

procedure TTclEditForm.BtnInsertTraceClick(Sender: TObject);
var
    trace: TTclTrace;
begin
//    trace := FTcl.BuildTrace( FNewDesigner );
//    TTclTrace(FNewDesigner.CreateComponent(TTclTrace, FTcl, 0, 0, 0, 0));
	trace := TTclTrace(FNewDesigner.CreateComponent(FTcl.BuildTrace, FTcl, 0, 0, 0, 0));
	if trace.Name = '' then
    	trace.Name := FNewDesigner.UniqueName(trace.ClassName);
    trace.Tcl := FTcl;
    with ListTraces do
    begin
    	items.addObject(trace.name, trace);
        itemIndex := items.indexOfObject(trace);
    end;
end;


procedure TTclEditForm.DeleteListItem(itemIndex: integer; items: TStrings; comp: TComponent);
begin
	if (itemIndex < 0) or (items.count <= 0) then exit;
	items.objects[itemIndex].free;
    items.delete(itemIndex);
    if items.count > 0 then
    begin
    	if itemIndex > 0 then itemIndex := itemIndex - 1;
        comp := TComponent(items.objects[itemIndex]);
    end;
//	RefreshLists;
    FNewDesigner.SelectComponent(comp);
    FNewDesigner.Modified;
end;


procedure TTclEditForm.BtnDeleteCommandClick(Sender: TObject);
begin
	with ListCommands do if ItemIndex >= 0 then
    	DeleteListItem(itemIndex, items, TTclCommand(items.objects[itemIndex]).Tcl);
	ListCommandsClick(nil);
end;

procedure TTclEditForm.BtnDeleteTraceClick(Sender: TObject);
begin
	with ListTraces do if ItemIndex >= 0 then
    	DeleteListItem(itemIndex, items, TTclTrace(items.objects[itemIndex]).Tcl);
end;

procedure TTclEditForm.FormDestroy(Sender: TObject);
begin
    if Assigned( FTcl ) then
    	FTcl.Designer := nil;
end;

function TTclEditForm.GetSelectedCommand: TTclCommand;
begin
	with ListCommands do
    begin
    	if itemIndex < 0 then result := nil
        else result := TTclCommand(items.objects[itemIndex]);
    end;
end;

procedure TTclEditForm.UpdateCommandButtons(enable: boolean);
begin
	BtnInsertSwitch.Enabled := enable;
    BtnDeleteSwitch.Enabled := enable;
    BtnInsertParam.Enabled := enable;
    BtnDeleteParam.Enabled := enable;
end;


procedure TTclEditForm.UpdateCommandInfo(cmd: TTclCommand);
begin
	if cmd = nil then
    begin
		LabelCommand.Caption := '';
        LabelSwitches.Caption := '';
        LabelParameters.Caption := '';
    end else
    begin
    	LabelCommand.Caption := format('%d Switches, %d Parameters', [ListSwitches.Items.Count, ListParams.Items.Count]);
		LabelSwitches.Caption := format('%s, %s', [Cmd.Name, Cmd.Command]);
        LabelParameters.Caption := LabelSwitches.Caption;
    end;

end;

procedure TTclEditForm.ListCommandsClick(Sender: TObject);
var
	x: integer;
	cmd: TTclCommand;
begin
	ListSwitches.Clear;
    ListParams.Clear;
    cmd := GetSelectedCommand;
    UpdateCommandButtons(cmd <> nil);
    if cmd <> nil then
    begin
 		BtnEmulate.Enabled := True;
		Select( cmd );
		with cmd do
	    begin
		    for x := 0 to SwitchCount - 1 do
				ListSwitches.items.addObject(Switches[x].Name, Switches[x]);
	        for x := 0 to ParamCount - 1 do
    	    	ListParams.items.addObject(Params[x].Name, Params[x]);
    	end;
	end else
    	BtnEmulate.Enabled := False;
    UpdateCommandInfo(cmd);

end;

procedure TTclEditForm.BtnInsertSwitchClick(Sender: TObject);
var
	cmd: TTclCommand;
    switch: TTclCmdSwitch;
begin
	cmd := GetSelectedCommand;
	if cmd = nil then exit;
    switch := TTclCmdSwitch(FNewDesigner.CreateComponent(cmd.BuildSwitch, cmd, 0, 0, 0, 0));
	if switch.name = '' then
	    switch.Name := FNewDesigner.UniqueName(switch.ClassName);
	switch.Command := cmd;
    with ListSwitches do
    begin
		items.addObject(switch.Name, switch);
    	itemIndex := items.indexOfObject(switch);
    end;
end;

procedure TTclEditForm.BtnDeleteSwitchClick(Sender: TObject);
begin
	with ListSwitches do if ItemIndex >= 0 then
        DeleteListItem(itemIndex, items, TTclCmdSwitch(items.objects[itemIndex]).Command);
end;

procedure TTclEditForm.ListTracesClick(Sender: TObject);
begin
	with ListTraces do
    	if itemIndex >= 0 then
			Select(TComponent(items.objects[itemIndex]));
end;

procedure TTclEditForm.ListSwitchesClick(Sender: TObject);
begin
	with ListSwitches do
    	if itemIndex >= 0 then
        	Select(TComponent(items.objects[itemIndex]));
end;

procedure TTclEditForm.ListParamsClick(Sender: TObject);
begin
	with ListParams do
    	if itemIndex >= 0 then
        	Select(TComponent(items.objects[itemIndex]));
end;

procedure TTclEditForm.BtnInsertParamClick(Sender: TObject);
var
	cmd: TTclCommand;
    param: TTclCmdParam;
begin
	cmd := GetSelectedCommand;
	if cmd = nil then exit;
	param :=  TTclCmdParam(FNewDesigner.CreateComponent(cmd.BuildParam, cmd, 0, 0, 0, 0));
	if param.name = '' then
	    param.Name := FNewDesigner.UniqueName(param.ClassName);
	param.Command := cmd;
    with ListParams do
    begin
    	items.addObject(param.Name, param);
        itemIndex := items.indexOfObject(param);
    end;
end;

procedure TTclEditForm.BtnDeleteParamClick(Sender: TObject);
begin
	with ListParams do if ItemIndex >= 0 then
        DeleteListItem(itemIndex, items, TTclCmdParam(items.objects[itemIndex]).Command);
end;

function TTclEditForm.CurrentHelpTopic: string;
begin
	if PageControl.ActivePage = SheetCommands then
    	result := 'TTclCommand'
    else if PageControl.ActivePage = SheetSwitches then
        result := 'TTclCmdSwitch'
    else if PageControl.ActivePage = SheetParams then
        result := 'TTclCmdParam'
    else if PageControl.ActivePage = SheetTraces then
        result := 'TTclTrace'
    else
    	result := 'TTcl';
end;

procedure TTclEditForm.DoHelp(ctx: string);
var
	prevHelp: string;
begin
    prevHelp := Application.HelpFile;
    try
    	Application.HelpFile := 'Tslc.hlp';
        Application.HelpJump(ctx);
    finally
    	Application.HelpFile := prevHelp;
    end;
end;

procedure TTclEditForm.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
	if Key = VK_F1 then
    	DoHelp(CurrentHelpTopic);
end;

procedure TTclEditForm.BtnHelpClick(Sender: TObject);
begin
	DoHelp(CurrentHelpTopic);
end;

procedure TTclEditForm.PageControlChange(Sender: TObject);
var
	p: pointer;
begin
	p := PageControl.ActivePage;
	BtnEmulate.Enabled := (p = SheetCommands) and (GetSelectedCommand <> nil);
    if p = SheetCommands then
    	ListCommands.OnClick(nil)
    else if p = SheetSwitches then
    	ListSwitches.OnClick(nil)
    else if p = SheetParams then
    	ListParams.OnClick(nil)
    else if p = SheetTraces then
    	ListTraces.OnClick(nil);
end;


//{$DEFINE BCB}
{$IFDEF BCB}
procedure TTclEditForm.BtnEmulateClick(Sender: TObject);
begin
end;
{$ELSE}
procedure TTclEditForm.BtnEmulateClick(Sender: TObject);
var
	frm: TEmulationForm;
    cmd: TTclCommand;
begin
	cmd := GetSelectedCommand;
    if cmd = nil then
    begin
    	BtnEmulate.Enabled := False;
        exit;
    end;
	frm := TEmulationForm.Create(Application);
    try
    	frm.Emulate(cmd, FNewDesigner);
    finally
    	frm.free;
    end;
end;
{$ENDIF}

procedure TTclEditForm.ListCommandsDblClick(Sender: TObject);
begin
	BtnEmulateClick(nil);
end;

procedure TTclEditForm.ListCommandsMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
	if not (ssDouble in Shift) then
		ListCommands.BeginDrag(False);
end;

procedure TTclEditForm.ListCommandsDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
	Accept := (Source = ListCommands) or
    	((Source is TListBox) and (TListBox(Source).Owner is TTclEditForm)
        	and (TListBox(Source).Name = 'ListCommands'));
end;

procedure TTclEditForm.ListCommandsDragDrop(Sender, Source: TObject; X,
  Y: Integer);
var
	srcIdx, destIdx: integer;
    cmd: TTclCommand;
begin
	if Source = ListCommands then
	with ListCommands do
    begin
    	srcIdx := ItemIndex;
        if srcIdx = -1 then exit;
    	destIdx := ItemAtPos(Point(X, Y), False);
        if destIdx >= items.Count then
        	destIdx := items.Count - 1;
    	FTcl.MoveCommand(srcIdx, destIdx);
		items.Move(srcIdx, destIdx);
    end  else
    with Source as TListBox do
	if ItemIndex >= 0 then
    begin
    	cmd := TTclCommand(items.objects[itemIndex]);
        if (cmd <> nil)then
        begin
        	if (cmd.Tcl.Owner <> self.FTcl.Owner) then
                MessageBox(0, 'Cannot move commands across modules', 'Tslc', MB_ICONSTOP)
            else
            begin
		        cmd.Tcl := self.FTcl;
		        (owner as TTclEditForm).RefreshLists;
    		    self.RefreshLists;
            end;
        end;
    end;
end;

procedure TTclEditForm.ListSwitchesDragDrop(Sender, Source: TObject; X,
  Y: Integer);
var
	srcIdx, destIdx: integer;
    cmd: TTclCommand;
begin
	with ListSwitches do
    begin
    	srcIdx := ItemIndex;
        if srcIdx = -1 then exit;
        cmd := TTclCmdSwitch(items.objects[srcIdx]).Command;
    	destIdx := ItemAtPos(Point(X, Y), False);
        if destIdx >= items.Count then
        	destIdx := items.Count - 1;
    	cmd.MoveSwitch(srcIdx, destIdx);
		items.Move(srcIdx, destIdx);
    end;
end;

procedure TTclEditForm.ListSwitchesDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
	Accept := Source = ListSwitches;
end;

procedure TTclEditForm.ListSwitchesMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
	if not (ssDouble in Shift) then
        ListSwitches.BeginDrag(False);
end;

procedure TTclEditForm.ListParamsDragDrop(Sender, Source: TObject; X,
  Y: Integer);
var
	srcIdx, destIdx: integer;
    cmd: TTclCommand;
begin
	with ListParams do
    begin
    	srcIdx := ItemIndex;
        if srcIdx = -1 then exit;
        cmd := TTclCmdParam(items.objects[srcIdx]).Command;
    	destIdx := ItemAtPos(Point(X, Y), False);
        if destIdx >= items.Count then
        	destIdx := items.Count - 1;
    	cmd.MoveParam(srcIdx, destIdx);
		items.Move(srcIdx, destIdx);
    end;
end;

procedure TTclEditForm.ListParamsDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
	Accept := (Source = ListParams);
end;

procedure TTclEditForm.ListParamsMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
	if not (ssDouble in Shift) then
    	ListParams.BeginDrag(False);
end;

procedure TTclEditForm.ListTracesDragDrop(Sender, Source: TObject; X,
  Y: Integer);
var
	srcIdx, destIdx: integer;
begin
	with ListTraces do
    begin
    	srcIdx := ItemIndex;
        if srcIdx = -1 then exit;
    	destIdx := ItemAtPos(Point(X, Y), False);
        if destIdx >= items.Count then
        	destIdx := items.Count - 1;
    	FTcl.MoveTrace(srcIdx, destIdx);
		items.Move(srcIdx, destIdx);
    end;
end;

procedure TTclEditForm.ListTracesDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
	Accept := Source = ListTraces;
end;

procedure TTclEditForm.ListTracesMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
	if not (ssDouble in Shift) then
    	ListTraces.BeginDrag(False);
end;

procedure TTclEditForm.BtnInsertChannelClick(Sender: TObject);
var
	x: integer;
begin
	with ListAvailChannels do
    for x:= 0 to  items.Count - 1 do
    	if Selected[x] then
            FTcl.InsertChannel(items.objects[x] as TTclChannel);
    RefreshChannelLists;
end;

procedure TTclEditForm.BtnRemoveChannelClick(Sender: TObject);
var
	x: integer;
begin
	with ListChannels do
    for x:= 0 to items.Count - 1 do
    	if Selected[x] then
        	FTcl.RemoveChannel(items.objects[x] as TTclChannel);
    RefreshChannelLists;
end;

procedure TTclEditForm.FormActivate(Sender: TObject);
begin
	RefreshChannelLists;
end;

end.
